# AppIntents Updates

> WWDC25 | iOS 26, macOS Tahoe, iPadOS 26, watchOS 26  
> 📖 https://developer.apple.com/documentation/AppIntents  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/228/

---

## 개요

App Intents 프레임워크는 앱의 핵심 기능을 Siri, Spotlight, 위젯, 단축어 등 시스템 전반에 통합합니다. WWDC25에서 **Visual Intelligence 지원**이 추가되었습니다.

---

## 주요 업데이트

### 1. Visual Intelligence 통합

사용자가 카메라나 화면에서 보고 있는 것에 대해 앱의 검색 기능을 연결할 수 있습니다.

```swift
import AppIntents

struct VisualSearchIntent: AppIntent {
    static var title: LocalizedStringResource = "시각적 검색"
    static var description = IntentDescription("이미지로 유사한 상품을 검색합니다")
    
    @Parameter(title: "이미지")
    var image: IntentFile
    
    func perform() async throws -> some IntentResult {
        let results = try await ProductSearch.findSimilar(image: image)
        return .result(value: results)
    }
}
```

### 2. Intelligent Shortcuts Actions

Apple Intelligence가 단축어에 통합되어 더 강력한 자동화가 가능합니다:

- **Writing Tools**: 텍스트 요약, 재작성
- **Image Playground**: 이미지 생성
- **"Use Model" 액션**: 온디바이스 AI 모델 직접 호출

### 3. Context-Aware Action Button

앱 컨텍스트에 따라 Action Button 동작을 동적으로 변경할 수 있습니다.

```swift
struct DynamicAction: AppIntent {
    static var title: LocalizedStringResource = "컨텍스트 액션"
    
    func perform() async throws -> some IntentResult {
        // 현재 앱 상태에 따라 다른 동작 수행
        return .result()
    }
}
```

### 4. 위젯 인터랙티비티 강화

- Control Center 빠른 제어
- 위젯 내에서 더 풍부한 상호작용
- CarPlay Live Activities 지원

---

## App Intents 기본 구조

```swift
import AppIntents

// 1. Intent 정의
struct OpenFavoriteIntent: AppIntent {
    static var title: LocalizedStringResource = "즐겨찾기 열기"
    static var openAppWhenRun: Bool = true
    
    @Parameter(title: "항목")
    var item: FavoriteItemEntity
    
    func perform() async throws -> some IntentResult & ProvidesDialog {
        // 실행 로직
        return .result(dialog: "\(item.name)을 열었습니다")
    }
}

// 2. Entity 정의
struct FavoriteItemEntity: AppEntity {
    static var typeDisplayRepresentation = TypeDisplayRepresentation(name: "즐겨찾기")
    static var defaultQuery = FavoriteItemQuery()
    
    var id: String
    var name: String
    
    var displayRepresentation: DisplayRepresentation {
        DisplayRepresentation(title: "\(name)")
    }
}

// 3. Query 정의
struct FavoriteItemQuery: EntityQuery {
    func entities(for identifiers: [String]) async throws -> [FavoriteItemEntity] {
        // 데이터 소스에서 엔티티 조회
        return FavoriteStore.shared.items(for: identifiers)
    }
    
    func suggestedEntities() async throws -> [FavoriteItemEntity] {
        return FavoriteStore.shared.topItems()
    }
}

// 4. AppShortcutsProvider
struct MyAppShortcuts: AppShortcutsProvider {
    static var appShortcuts: [AppShortcut] {
        AppShortcut(
            intent: OpenFavoriteIntent(),
            phrases: ["Open my favorite in \(.applicationName)"],
            shortTitle: "즐겨찾기 열기",
            systemImageName: "star.fill"
        )
    }
}
```

---

## Visual Intelligence 검색 통합

```swift
// 앱의 비주얼 검색을 Visual Intelligence에 등록
struct VisualSearchProvider: AppIntent {
    static var title: LocalizedStringResource = "비주얼 검색"
    
    @Parameter(title: "검색 이미지")
    var searchImage: IntentFile
    
    func perform() async throws -> some IntentResult {
        // 이미지 분석 및 유사 항목 검색
        let matches = try await catalog.findMatches(for: searchImage)
        return .result(value: matches)
    }
}
```

---

## 관련 세션

- [What's new in App Intents (228)](https://developer.apple.com/videos/play/wwdc2025/228/)
- [Bring your app to Siri (231)](https://developer.apple.com/videos/play/wwdc2025/231/)
