# Foundation AttributedString Updates

> WWDC25 | iOS 26, macOS Tahoe  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/256/  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/258/

---

## 개요

iOS 26에서 `AttributedString`이 SwiftUI의 `TextView`와 깊이 통합되어 **리치 텍스트 편집**이 가능해졌습니다. Markdown 파싱 기능도 강화되었습니다.

---

## SwiftUI에서 리치 텍스트 편집

### TextEditor + AttributedString

```swift
import SwiftUI

struct RichTextView: View {
    @State private var text = AttributedString("Hello, World!")
    
    var body: some View {
        // iOS 26: TextEditor가 AttributedString 바인딩 지원
        TextEditor(text: $text)
    }
}
```

### TextView (신규)

```swift
struct ContentView: View {
    @State private var attributedText: AttributedString = {
        var str = AttributedString("볼드 텍스트")
        str.font = .body.bold()
        return str
    }()
    
    var body: some View {
        // 새로운 TextView - 리치 텍스트 편집 전용
        TextView(text: $attributedText)
    }
}
```

---

## Markdown → AttributedString 변환

```swift
// Markdown 문자열을 AttributedString으로 변환
let markdown = "**볼드** 그리고 *이탤릭* 텍스트"
let attributed = try AttributedString(markdown: markdown)

// presentationIntent를 사용하여 커스텀 폰트 매핑
func applyCustomStyles(_ attributedString: AttributedString) -> AttributedString {
    var result = attributedString
    for run in result.runs {
        if let intent = run.presentationIntent {
            for component in intent.components {
                switch component.kind {
                case .header(let level):
                    // 헤딩 레벨에 따른 폰트 설정
                    let range = run.range
                    result[range].font = headerFont(level: level)
                default:
                    break
                }
            }
        }
    }
    return result
}
```

---

## AttributedString 속성 조작

### 기본 스타일링

```swift
var str = AttributedString("스타일 텍스트")

// 폰트 설정
str.font = .system(size: 16, weight: .bold)

// 색상
str.foregroundColor = .blue

// 밑줄
str.underlineStyle = .single

// 취소선
str.strikethroughStyle = .single
str.strikethroughColor = .red
```

### 부분 범위 스타일링

```swift
var str = AttributedString("Hello, SwiftUI World!")

if let range = str.range(of: "SwiftUI") {
    str[range].font = .headline
    str[range].foregroundColor = .accentColor
}
```

### 링크

```swift
var str = AttributedString("Apple 공식 문서를 확인하세요")
if let range = str.range(of: "Apple 공식 문서") {
    str[range].link = URL(string: "https://developer.apple.com")
}
```

---

## Runs 순회

```swift
let attributed = try AttributedString(markdown: "**Bold** and *italic* text")

for run in attributed.runs {
    print("Text: \(attributed[run.range])")
    print("Bold: \(run.font?.isBold ?? false)")
    print("---")
}
```

---

## 커스텀 속성 정의

```swift
enum HighlightColorAttribute: AttributedStringKey {
    typealias Value = Color
    static let name = "highlightColor"
}

extension AttributeScopes {
    struct MyAppAttributes: AttributeScope {
        let highlightColor: HighlightColorAttribute
    }
    
    var myApp: MyAppAttributes.Type { MyAppAttributes.self }
}

// 사용
var str = AttributedString("하이라이트")
str.myApp.highlightColor = .yellow
```

---

## 관련 세션

- [What's new in SwiftUI (256)](https://developer.apple.com/videos/play/wwdc2025/256/)
- [Code-along: Cook up a rich text experience (258)](https://developer.apple.com/videos/play/wwdc2025/258/)
