# FoundationModels: Using on-device LLM in your app

> WWDC25 | iOS 26, macOS Tahoe, iPadOS 26, visionOS 26  
> 📖 https://developer.apple.com/documentation/FoundationModels  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/286/  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/301/

---

## 개요

Foundation Models 프레임워크는 Apple Intelligence를 구동하는 **온디바이스 대형 언어 모델(LLM)**에 직접 접근할 수 있는 Swift API를 제공합니다.

- **온디바이스 실행**: 데이터가 기기를 떠나지 않아 프라이버시 보장
- **오프라인 사용 가능**: 네트워크 연결 없이도 동작
- **앱 크기 증가 없음**: OS에 내장되어 있음
- **무료**: API 호출당 비용 없음
- **30억 파라미터** 모델로 요약, 추출, 분류 등에 최적화

---

## 기본 사용법

### Hello World

```swift
import FoundationModels

let session = LanguageModelSession()
let response = try await session.respond(to: "미국의 주를 나열해주세요.")
print(response.content)
```

### 모델 가용성 확인

```swift
let systemModel = SystemLanguageModel.default
guard systemModel.isAvailable else {
    print("Foundation Models을 사용할 수 없습니다")
    return
}
```

### Instructions와 Prompt 분리

```swift
let session = LanguageModelSession(
    instructions: """
    당신은 친절한 바리스타입니다.
    짧고 명확하게 답변하세요.
    """
)

let response = try await session.respond(to: "오늘의 추천 커피는?")
```

---

## Guided Generation (@Generable)

모델 출력을 Swift 구조체로 자동 파싱하는 핵심 기능입니다.

### 기본 사용

```swift
import FoundationModels

@Generable
struct Itinerary {
    var destination: String
    var activities: [Activity]
}

@Generable
struct Activity {
    var name: String
    var duration: String
    var description: String
}

let session = LanguageModelSession()
let itinerary: Itinerary = try await session.respond(
    to: "제주도 2박 3일 여행 일정을 만들어주세요",
    generating: Itinerary.self
).content
```

### @Guide 매크로로 힌트 제공

```swift
@Generable
struct NPC {
    @Guide(description: "캐릭터 이름, 2-3 단어")
    var name: String
    
    @Guide(range: 1...100)
    var level: Int
}
```

### Enum 지원

```swift
@Generable
enum Encounter {
    case coffeeOrder(order: String)
    case complaint(reason: String)
}
```

### Constrained Decoding

Guided Generation은 **제약된 디코딩(Constrained Decoding)**을 사용합니다:
- 모델이 토큰을 생성할 때 스키마에 맞지 않는 토큰을 마스킹
- 구조적 오류(잘못된 필드 이름 등) 방지
- 수동 파싱 불필요

---

## 스트리밍

```swift
let stream = session.streamResponse(
    to: "여행 계획을 세워주세요",
    generating: Itinerary.self
)

for try await partial in stream {
    // partial.content로 부분 결과 접근
    updateUI(with: partial)
}
```

---

## Tool Calling

모델이 외부 데이터에 접근할 수 있도록 커스텀 도구를 정의합니다.

```swift
struct GetWeatherTool: Tool {
    let name = "getWeather"
    let description = "특정 도시의 현재 날씨를 가져옵니다."
    
    @Generable
    struct Arguments {
        var city: String
    }
    
    func call(arguments: Arguments) async throws -> ToolOutput {
        let weather = await WeatherService.fetch(city: arguments.city)
        return ToolOutput(weather.description)
    }
}

let session = LanguageModelSession(tools: [GetWeatherTool()])
```

특징:
- 도구는 한 요청 내에서 **여러 번 호출** 가능
- **병렬 호출** 지원
- 도구에 **상태 저장** 가능

---

## 세션 관리

```swift
// 세션은 대화 기록(Transcript)을 유지
let session = LanguageModelSession()

// 첫 번째 요청
let r1 = try await session.respond(to: "나는 커피를 좋아해")

// 두 번째 요청 - 이전 컨텍스트를 기억함
let r2 = try await session.respond(to: "그럼 뭘 추천해줄래?")

// Transcript 접근
print(session.transcript)
```

### Prewarm (사전 로드)

```swift
// LLM 리소스를 미리 메모리에 로드
try await LanguageModelSession.prewarm()
```

---

## Xcode Playground에서 실험

```swift
import Playgrounds
import FoundationModels

#Playground {
    let session = LanguageModelSession()
    let response = try await session.respond(to: "Swift의 장점 3가지")
    print(response.content)
}
```

---

## 모델 제한 사항

| 가능 | 제한 |
|------|------|
| 요약, 분류, 추출 | ❌ 복잡한 추론 |
| 텍스트 생성/수정 | ❌ 수학 계산 |
| 다중 턴 대화 | ❌ 코드 생성 |
| 구조화된 출력 | ❌ 광범위한 세계 지식 |
| 태그 생성 | ❌ 학습 날짜 이후 지식 |

- **토큰 제한**: 입출력 합계 4096 토큰
- **속성 순서 중요**: 다른 속성에 의존하는 속성(예: 요약)은 마지막에 배치
- **영어에서 최적 성능**: Instructions는 영어로 작성 권장

---

## 관련 세션

- [Meet the Foundation Models framework (286)](https://developer.apple.com/videos/play/wwdc2025/286/)
- [Deep dive into the Foundation Models framework (301)](https://developer.apple.com/videos/play/wwdc2025/301/)
- [Code-along: Bring on-device AI to your app (259)](https://developer.apple.com/videos/play/wwdc2025/259/)
- [Explore prompt design & safety (248)](https://developer.apple.com/videos/play/wwdc2025/248/)
