# Implementing Assistive Access in iOS

> WWDC25 | iOS 26  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/246/  
> 📖 https://developer.apple.com/documentation/Accessibility

---

## 개요

Assistive Access는 인지 장애가 있는 사용자를 위한 맞춤형 iOS 경험입니다. iOS 26에서 개발자가 자신의 앱을 Assistive Access 모드에 맞게 최적화할 수 있는 새로운 API가 제공됩니다.

---

## Assistive Access란?

- **간소화된 인터페이스**: 큰 버튼, 단순한 레이아웃
- **핵심 기능에 집중**: 불필요한 복잡성 제거
- **보호자/가족 설정**: 사용할 앱과 기능을 선택 가능
- iPhone의 주요 앱(전화, 메시지, 카메라, 음악 등) 지원

---

## SwiftUI에서 Assistive Access Scene

### AssistiveAccessScene (신규)

```swift
import SwiftUI

@main
struct MyApp: App {
    var body: some Scene {
        // 일반 모드
        WindowGroup {
            ContentView()
        }
        
        // Assistive Access 모드에서의 간소화된 뷰
        AssistiveAccessScene {
            SimplifiedContentView()
        }
    }
}
```

### 간소화된 뷰 설계

```swift
struct SimplifiedContentView: View {
    var body: some View {
        VStack(spacing: 24) {
            // 큰 버튼, 명확한 아이콘
            Button(action: takePhoto) {
                Label("사진 찍기", systemImage: "camera.fill")
                    .font(.title)
                    .frame(maxWidth: .infinity)
                    .padding()
            }
            .buttonStyle(.borderedProminent)
            .controlSize(.large)
            
            Button(action: viewPhotos) {
                Label("사진 보기", systemImage: "photo.fill")
                    .font(.title)
                    .frame(maxWidth: .infinity)
                    .padding()
            }
            .buttonStyle(.borderedProminent)
            .controlSize(.large)
        }
        .padding(32)
    }
}
```

---

## Assistive Access 모드 감지

```swift
import SwiftUI

struct AdaptiveView: View {
    @Environment(\.assistiveAccessEnabled) var isAssistiveAccess
    
    var body: some View {
        if isAssistiveAccess {
            SimplifiedLayout()
        } else {
            FullLayout()
        }
    }
}
```

---

## 디자인 가이드라인

### 레이아웃

- **큰 탭 타겟**: 최소 60×60 포인트
- **단순한 네비게이션**: 깊은 계층 피하기
- **명확한 라벨**: 아이콘과 텍스트 함께 사용
- **여백 충분히**: 요소 간 간격 넓게

### 콘텐츠

- **쉬운 언어 사용**: 짧고 명확한 문장
- **핵심 기능만 노출**: 고급 설정은 숨기기
- **시각적 피드백**: 모든 액션에 명확한 응답

### 접근성

```swift
// 접근성 라벨 명시
Button("전화") { }
    .accessibilityLabel("전화 걸기")
    .accessibilityHint("친구에게 전화합니다")

// Dynamic Type 지원 필수
Text("안녕하세요")
    .font(.title)
    .dynamicTypeSize(...DynamicTypeSize.accessibility5)
```

---

## 관련 세션

- [Build accessible experiences for Assistive Access (246)](https://developer.apple.com/videos/play/wwdc2025/246/)
