# Implementing Visual Intelligence in iOS

> WWDC25 | iOS 26  
> 📖 https://developer.apple.com/documentation/VisualIntelligence  
> 📖 https://developer.apple.com/documentation/VisualIntelligence/integrating-your-app-with-visual-intelligence

---

## 개요

Visual Intelligence가 iPhone **화면 콘텐츠**로 확장됩니다. 사용자가 화면에서 보고 있는 것에 대해 검색하고, 질문하고, 행동을 취할 수 있습니다. 개발자는 App Intents를 통해 앱의 검색 기능을 Visual Intelligence에 통합할 수 있습니다.

---

## Visual Intelligence 동작 방식

### 사용자 경험

1. 스크린샷 버튼을 눌러 Visual Intelligence 활성화
2. 화면 콘텐츠 분석
3. ChatGPT에게 질문하거나 Google/Etsy 등에서 검색
4. 특정 객체를 하이라이트하여 개별 검색 가능

### 지원 범위

- **카메라**: 주변 사물과 장소 인식 (기존)
- **화면 콘텐츠**: 앱에서 보고 있는 내용 검색 (신규)
- **이벤트 인식**: 이벤트 정보를 캘린더에 자동 추가

---

## 앱 통합: Visual Search Provider

```swift
import AppIntents
import VisualIntelligence

// 비주얼 검색 인텐트 정의
struct ProductVisualSearch: AppIntent {
    static var title: LocalizedStringResource = "상품 비주얼 검색"
    static var description = IntentDescription(
        "이미지와 유사한 상품을 앱에서 검색합니다"
    )
    
    @Parameter(title: "검색 이미지")
    var image: IntentFile
    
    static var parameterSummary: some ParameterSummary {
        Summary("\\(\\.$image)와 유사한 상품 검색")
    }
    
    func perform() async throws -> some IntentResult & ReturnsValue<[ProductEntity]> {
        // 이미지 분석 및 유사 상품 검색
        let imageData = try image.data
        let results = try await ProductCatalog.shared.searchByImage(imageData)
        
        return .result(value: results.map { ProductEntity(from: $0) })
    }
}
```

---

## Visual Intelligence 검색 결과 제공

```swift
// 검색 결과 엔티티
struct ProductEntity: AppEntity {
    static var typeDisplayRepresentation = TypeDisplayRepresentation(name: "상품")
    static var defaultQuery = ProductQuery()
    
    var id: String
    var name: String
    var price: String
    var imageURL: URL?
    
    var displayRepresentation: DisplayRepresentation {
        DisplayRepresentation(
            title: "\(name)",
            subtitle: "\(price)",
            image: imageURL.map { .init(url: $0) }
        )
    }
}
```

---

## 캘린더 이벤트 인식 통합

Visual Intelligence는 이벤트를 자동으로 인식합니다:

- 날짜, 시간, 장소 자동 추출
- 캘린더 이벤트 미리 채우기
- Apple Intelligence가 데이터를 프라이빗하게 처리

---

## 모범 사례

1. **고품질 이미지 매칭**: 다양한 각도와 조명에서 작동하도록
2. **빠른 응답**: Visual Intelligence는 즉각적인 결과를 기대
3. **관련성 높은 결과**: 상위 결과의 품질이 중요
4. **딥링크 제공**: 검색 결과에서 앱 내 해당 항목으로 직접 이동
5. **프라이버시**: 이미지 데이터를 불필요하게 저장하지 않기

---

## 관련 문서

- [Visual Intelligence 공식 문서](https://developer.apple.com/documentation/VisualIntelligence)
- [앱 통합 가이드](https://developer.apple.com/documentation/VisualIntelligence/integrating-your-app-with-visual-intelligence)
- [Apple Intelligence 개요](https://developer.apple.com/apple-intelligence/)
