# MapKit GeoToolbox & PlaceDescriptors

> WWDC25 | iOS 26, macOS Tahoe  
> 📖 https://developer.apple.com/documentation/GeoToolbox  
> 📖 https://developer.apple.com/documentation/MapKit  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/263/

---

## 개요

iOS 26에서 새로운 **GeoToolbox** 프레임워크가 도입되어, 지도 좌표에 대한 장소 설명자(Place Descriptor) 정보를 제공합니다.

---

## GeoToolbox 프레임워크

### PlaceDescriptor

좌표를 사람이 읽을 수 있는 장소 정보로 변환합니다.

```swift
import GeoToolbox
import CoreLocation

// 좌표로부터 장소 설명 가져오기
let coordinate = CLLocationCoordinate2D(latitude: 37.7749, longitude: -122.4194)

let descriptor = try await PlaceDescriptor(coordinate: coordinate)

print(descriptor.name)          // "San Francisco"
print(descriptor.locality)      // 도시명
print(descriptor.administrativeArea) // 주/도
print(descriptor.country)       // 국가
```

### 주요 속성

| 속성 | 설명 |
|------|------|
| `name` | 장소 이름 |
| `locality` | 도시/구 |
| `subLocality` | 동/읍/면 |
| `administrativeArea` | 시/도 |
| `country` | 국가 |
| `postalCode` | 우편번호 |
| `thoroughfare` | 도로명 |

---

## MapKit 업데이트

### 새로운 기능들

```swift
import MapKit
import SwiftUI

struct MapView: View {
    @State private var position: MapCameraPosition = .automatic
    
    var body: some View {
        Map(position: $position) {
            // 마커 추가
            Marker("서울타워", coordinate: CLLocationCoordinate2D(
                latitude: 37.5512, longitude: 126.9882
            ))
            
            // 어노테이션
            Annotation("카페", coordinate: cafeLocation) {
                Image(systemName: "cup.and.saucer.fill")
                    .foregroundStyle(.brown)
            }
        }
        .mapStyle(.standard(elevation: .realistic))
    }
}
```

### Place Descriptor와 MapKit 연동

```swift
// 탭한 위치의 장소 정보 표시
struct InteractiveMapView: View {
    @State private var selectedPlace: PlaceDescriptor?
    
    var body: some View {
        Map {
            // 맵 콘텐츠
        }
        .onMapTap { coordinate in
            Task {
                selectedPlace = try await PlaceDescriptor(coordinate: coordinate)
            }
        }
        .sheet(item: $selectedPlace) { place in
            PlaceDetailView(place: place)
        }
    }
}
```

---

## 지오코딩 vs PlaceDescriptor

| 기능 | CLGeocoder | PlaceDescriptor |
|------|-----------|----------------|
| 방향 | 좌표 ↔ 주소 양방향 | 좌표 → 설명 단방향 |
| 데이터 | 상세 주소 | 장소 컨텍스트 |
| 용도 | 정확한 주소 변환 | 위치의 의미적 설명 |
| 네트워크 | 필요 | 온디바이스 가능 |

---

## 관련 세션

- [What's new in MapKit (263)](https://developer.apple.com/videos/play/wwdc2025/263/)
