# StoreKit Updates

> WWDC25 | iOS 26, macOS Tahoe  
> 📖 https://developer.apple.com/documentation/StoreKit  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/236/

---

## 개요

StoreKit이 WWDC25에서 업데이트되어 인앱 구매, 구독 관리, 영수증 검증 등이 개선되었습니다.

---

## StoreKit Views (SwiftUI 통합)

### SubscriptionStoreView

```swift
import StoreKit
import SwiftUI

struct PaywallView: View {
    var body: some View {
        SubscriptionStoreView(groupID: "com.myapp.premium") {
            // 커스텀 마케팅 콘텐츠
            VStack {
                Image("premium-banner")
                    .resizable()
                    .aspectRatio(contentMode: .fit)
                
                Text("프리미엄으로 업그레이드")
                    .font(.title)
                
                Text("모든 기능을 잠금 해제하세요")
                    .foregroundStyle(.secondary)
            }
        }
        .subscriptionStoreButtonLabel(.multiline)
        .storeButton(.visible, for: .restorePurchases)
    }
}
```

### ProductView

```swift
struct StoreView: View {
    let productIDs = ["com.myapp.feature1", "com.myapp.feature2"]
    
    var body: some View {
        ForEach(productIDs, id: \.self) { id in
            ProductView(id: id) {
                // 커스텀 아이콘
                Image(systemName: "star.fill")
            }
        }
        .productViewStyle(.large)
    }
}
```

---

## Transaction 관리

### 최신 트랜잭션 확인

```swift
// 특정 상품의 최신 트랜잭션
let result = try await Transaction.latest(for: "com.myapp.premium")

if let transaction = result {
    // 유효한 구매 확인
    if transaction.revocationDate == nil {
        // 기능 잠금 해제
        unlockPremium()
    }
}
```

### 트랜잭션 리스너

```swift
// 앱 시작 시 트랜잭션 업데이트 모니터링
Task {
    for await result in Transaction.updates {
        guard case .verified(let transaction) = result else { continue }
        
        // 구매 상태 업데이트
        await updatePurchaseStatus(transaction)
        
        // 트랜잭션 완료 처리
        await transaction.finish()
    }
}
```

---

## 구독 상태 관리

```swift
// 구독 그룹 상태 확인
let statuses = try await Product.SubscriptionInfo.status(
    for: "com.myapp.premium"
)

for status in statuses {
    switch status.state {
    case .subscribed:
        print("활성 구독")
    case .expired:
        print("만료됨")
    case .inBillingRetryPeriod:
        print("결제 재시도 중")
    case .inGracePeriod:
        print("유예 기간")
    case .revoked:
        print("취소됨")
    default:
        break
    }
}
```

---

## App Store Server API 업데이트

### 영수증 검증 (서버 사이드)

```swift
// StoreKit 2의 JWS 검증
let result = try await AppTransaction.shared

switch result {
case .verified(let appTransaction):
    print("앱 ID: \(appTransaction.appID)")
    print("번들 ID: \(appTransaction.bundleID)")
case .unverified(_, let error):
    print("검증 실패: \(error)")
}
```

---

## 테스트 (StoreKit Testing in Xcode)

```swift
// StoreKit Configuration 파일로 로컬 테스트
// 1. File > New > StoreKit Configuration File
// 2. 상품/구독 정의
// 3. Scheme에서 StoreKit Configuration 선택

// 단위 테스트
@Test func testPurchase() async throws {
    let session = try SKTestSession(configurationFileNamed: "Products")
    session.disableDialogs = true
    session.clearTransactions()
    
    // 구매 시뮬레이션
    let product = try await Product.products(for: ["com.myapp.premium"]).first!
    let result = try await product.purchase()
    
    // 결과 확인
    guard case .success(let verification) = result,
          case .verified(let transaction) = verification else {
        Issue.record("구매 실패")
        return
    }
    
    await transaction.finish()
}
```

---

## 관련 세션

- [What's new in StoreKit (236)](https://developer.apple.com/videos/play/wwdc2025/236/)
