# Swift Charts 3D Visualization

> WWDC25 | iOS 26, macOS Tahoe, visionOS 26  
> 📖 https://developer.apple.com/documentation/Charts  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/313/

---

## 개요

Swift Charts에 **Chart3D**가 추가되어 3D 데이터 시각화가 가능해졌습니다. RealityKit 지원으로 인터랙티브한 3D 차트를 SwiftUI에서 직접 만들 수 있습니다.

---

## 기본 사용법: 2D → 3D

### 2D 차트

```swift
import Charts

Chart(penguinData) { penguin in
    PointMark(
        x: .value("날개 길이", penguin.flipperLength),
        y: .value("무게", penguin.weight)
    )
    .foregroundStyle(by: .value("종", penguin.species))
}
```

### 3D 차트로 변환

```swift
import Charts

// Chart → Chart3D, Z 값 추가
Chart3D(penguinData) { penguin in
    PointMark(
        x: .value("날개 길이", penguin.flipperLength),
        y: .value("무게", penguin.weight),
        z: .value("부리 길이", penguin.beakLength)  // Z축 추가
    )
    .foregroundStyle(by: .value("종", penguin.species))
}
```

---

## 지원 Mark 타입

| Mark | 용도 |
|------|------|
| `PointMark` | 3D 산점도 |
| `RuleMark` | 3D 규칙선 |
| `RectangleMark` | 3D 직육면체 |
| `SurfacePlot` | 수학적 표면 플롯 (신규) |

---

## SurfacePlot

2변수 함수를 3D 표면으로 시각화합니다.

```swift
Chart3D {
    // y = f(x, z) 형태의 함수 표면
    SurfacePlot(x: "X", y: "Y", z: "Z") { x, z in
        sin(x) * cos(z)  // 수학적 표현식
    }
    .foregroundStyle(.blue.gradient)
}
```

### 복합 차트: 데이터 + 표면

```swift
Chart3D {
    // 데이터 포인트
    ForEach(dataPoints) { point in
        PointMark(
            x: .value("X", point.x),
            y: .value("Y", point.y),
            z: .value("Z", point.z)
        )
    }
    
    // 회귀 표면
    SurfacePlot(x: "X", y: "Y", z: "Z") { x, z in
        regressionModel(x, z)
    }
    .foregroundStyle(.gray.opacity(0.7))
}
```

---

## 카메라 및 커스터마이징

### 시점 설정

```swift
Chart3D(data) { item in
    PointMark(
        x: .value("X", item.x),
        y: .value("Y", item.y),
        z: .value("Z", item.z)
    )
}
// 카메라 시점 설정
.chart3DPose(.init(azimuth: .degrees(45), elevation: .degrees(30)))
```

### 축 스케일

```swift
Chart3D(irisData) { iris in
    PointMark(
        x: .value("꽃잎 길이", iris.petalLength),
        y: .value("꽃잎 너비", iris.petalWidth),
        z: .value("꽃받침 길이", iris.sepalLength)
    )
}
.chartXScale(domain: 1...8, range: -1.5...1.5)
.chartYScale(domain: 0...3, range: -0.5...0.5)
.chartZScale(domain: 4...8, range: -0.5...0.5)  // 신규
```

### 축 라벨

```swift
.chartXAxisLabel("꽃잎 길이 (cm)")
.chartYAxisLabel("꽃잎 너비 (cm)")
.chartZAxisLabel("꽃받침 길이 (cm)")
```

---

## 인터랙션

3D 차트는 기본적으로 인터랙티브합니다:

- **회전**: 드래그 제스처로 자유롭게 회전
- **측면 보기**: 특정 각도로 회전하면 2D 차트처럼 보임
- **visionOS**: 공간적 인터랙션 지원

---

## 카메라 프로젝션

두 가지 카메라 프로젝션을 지원합니다:

```swift
// 원근 투영 (기본값)
.chart3DProjection(.perspective)

// 직교 투영
.chart3DProjection(.orthographic)
```

---

## 관련 세션

- [Bring Swift Charts to the third dimension (313)](https://developer.apple.com/videos/play/wwdc2025/313/)
