# Swift Concurrency Updates

> WWDC25 | Swift 6.2  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/250/

---

## 개요

Swift 6.2에서 동시성 모델이 크게 간소화되었습니다. 단일 스레드 코드 작성이 더 쉬워지고, MainActor 기본 설정이 가능해졌습니다.

---

## 주요 변경: MainActor 기본 설정

### 모듈/파일 레벨 MainActor

```swift
// 파일 전체를 MainActor에서 실행하도록 설정
// Swift 6.2에서 모듈 또는 개별 파일 단위로 설정 가능

// 이전: 각 타입마다 @MainActor 표기 필요
@MainActor
class ViewModel { }

@MainActor
struct ContentView: View { }

// 이후: 모듈 설정으로 기본값 적용
// 별도 어노테이션 불필요
class ViewModel { }  // 자동으로 MainActor
struct ContentView: View { }  // 자동으로 MainActor
```

---

## nonisolated 키워드 개선

```swift
// MainActor 기본 설정 시, 
// 백그라운드 작업이 필요한 경우 nonisolated 명시

class DataProcessor {
    // 기본: MainActor에서 실행
    func updateUI() {
        // UI 업데이트 코드
    }
    
    // 명시적으로 비격리
    nonisolated func processData(_ data: Data) async -> Result {
        // 백그라운드에서 실행
        return heavyComputation(data)
    }
}
```

---

## Strict Concurrency 간소화

### Swift 6 vs Swift 6.2

```swift
// Swift 6: 엄격한 동시성 검사로 많은 어노테이션 필요
@MainActor
class MyViewModel: ObservableObject {
    @MainActor @Published var items: [Item] = []
    
    @MainActor
    func loadItems() async {
        // ...
    }
}

// Swift 6.2: 파일/모듈 기본 설정으로 간소화
class MyViewModel: ObservableObject {
    @Published var items: [Item] = []
    
    func loadItems() async {
        // 자동으로 MainActor
    }
}
```

---

## Task 그룹 개선

```swift
// TaskGroup의 새로운 패턴
func fetchAllData() async throws -> [Item] {
    try await withThrowingTaskGroup(of: Item.self) { group in
        for id in itemIDs {
            group.addTask {
                try await fetchItem(id: id)
            }
        }
        
        return try await group.reduce(into: []) { $0.append($1) }
    }
}
```

---

## Actor 관련 업데이트

### Custom Actor Executors

```swift
actor DatabaseManager {
    // 커스텀 직렬 디스패치 큐에서 실행
    nonisolated var unownedExecutor: UnownedSerialExecutor {
        queue.asUnownedSerialExecutor()
    }
    
    private let queue = DispatchSerialQueue(label: "database")
    
    func query(_ sql: String) async -> [Row] {
        // queue에서 실행됨
        return database.execute(sql)
    }
}
```

---

## Sendable 관련 개선

```swift
// 더 많은 타입이 자동으로 Sendable
// Swift 6.2에서 컴파일러 추론 개선

struct Configuration: Sendable {
    let name: String
    let count: Int
    // 모든 저장 프로퍼티가 Sendable이면 자동으로 Sendable
}
```

---

## C++, Java, JavaScript 상호 운용성

Swift 6.2에서 다른 언어와의 상호 운용성이 강화되었습니다:

- **C++**: 개선된 브릿징
- **Java**: Android/JVM 상호 운용
- **JavaScript**: WebAssembly 지원

### WebAssembly 지원 (신규)

```swift
// Swift 코드를 WebAssembly로 컴파일
// 오픈소스 커뮤니티와 협력하여 지원
```

---

## 관련 세션

- [What's new in Swift (250)](https://developer.apple.com/videos/play/wwdc2025/250/)
