# Swift InlineArray & Span

> WWDC25 | Swift 6.2  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/250/  
> 📖 Swift Evolution: SE-0453 (InlineArray), SE-0447 (Span)

---

## 개요

Swift 6.2에서 **InlineArray**와 **Span** 두 가지 새로운 저수준 메모리 타입이 도입되었습니다. 성능이 중요한 코드에서 안전하면서도 효율적인 메모리 접근을 제공합니다.

---

## InlineArray (SE-0453)

고정 크기 배열을 **인라인(스택)**에 저장하는 타입입니다. 힙 할당 없이 고정 개수의 요소를 다룹니다.

### 기본 사용

```swift
// 3개의 Int를 인라인으로 저장
var rgb: InlineArray<3, Int> = [255, 128, 0]

// 인덱스 접근
print(rgb[0])  // 255
rgb[2] = 64

// 반복
for value in rgb {
    print(value)
}
```

### 초기화

```swift
// 반복 값으로 초기화
let zeros = InlineArray<4, Double>(repeating: 0.0)

// 클로저로 초기화
let indices = InlineArray<5, Int> { index in
    index * 2  // [0, 2, 4, 6, 8]
}
```

### Array와의 차이점

| 특성 | `Array<T>` | `InlineArray<N, T>` |
|------|-----------|---------------------|
| 저장 위치 | 힙 (Heap) | 인라인 (Stack) |
| 크기 | 가변 | 컴파일 타임 고정 |
| 복사 비용 | COW (참조 기반) | 값 전체 복사 |
| 용도 | 범용 | 성능 중요 고정 크기 |

### 활용 사례

```swift
// 3D 벡터
struct Vector3D {
    var components: InlineArray<3, Double>
    
    var x: Double { components[0] }
    var y: Double { components[1] }
    var z: Double { components[2] }
    
    init(x: Double, y: Double, z: Double) {
        components = [x, y, z]
    }
}

// 행렬
struct Matrix4x4 {
    var elements: InlineArray<16, Float>
}

// 색상 (RGBA)
struct Color {
    var channels: InlineArray<4, UInt8>  // R, G, B, A
}
```

---

## Span (SE-0447)

메모리의 **연속적인 영역에 대한 비소유 뷰(non-owning view)**입니다. 복사 없이 메모리를 안전하게 참조합니다.

### 기본 사용

```swift
let array = [1, 2, 3, 4, 5]

// Array로부터 Span 생성
let span = array.span

// 읽기 전용 접근
print(span[0])       // 1
print(span.count)    // 5

// 슬라이싱
let sub = span[1..<3]  // [2, 3]에 대한 뷰
```

### MutableSpan

```swift
var data = [10, 20, 30, 40, 50]

// 수정 가능한 Span
data.withMutableSpan { mutableSpan in
    mutableSpan[0] = 100
    mutableSpan[2] = 300
}
// data는 이제 [100, 20, 300, 40, 50]
```

### Span vs UnsafeBufferPointer

| 특성 | `Span<T>` | `UnsafeBufferPointer<T>` |
|------|----------|--------------------------|
| 안전성 | 안전함 (수명 관리) | 안전하지 않음 |
| 바운드 체크 | 런타임 체크 | 없음 |
| 용도 | 범용 메모리 뷰 | 저수준 상호 운용 |
| 소유권 | 비소유 (대여) | 비소유 |

### 함수에서 Span 사용

```swift
// Span을 받아서 처리하는 함수
func sum(_ values: Span<Int>) -> Int {
    var total = 0
    for value in values {
        total += value
    }
    return total
}

// Array, InlineArray 등 다양한 소스에서 호출 가능
let array = [1, 2, 3]
let result = sum(array.span)

var inline: InlineArray<3, Int> = [4, 5, 6]
let result2 = sum(inline.span)
```

---

## InlineArray + Span 연동

```swift
// InlineArray의 데이터를 Span으로 전달
var pixels: InlineArray<1024, UInt8> = .init(repeating: 0)

func processPixels(_ data: Span<UInt8>) {
    for i in 0..<data.count {
        // 읽기 전용 처리
    }
}

processPixels(pixels.span)
```

---

## 성능 고려사항

```swift
// ✅ 좋은 사용: 작은 고정 크기 데이터
struct Vertex {
    var position: InlineArray<3, Float>   // 12 bytes, 스택에
    var normal: InlineArray<3, Float>     // 12 bytes, 스택에
    var texCoord: InlineArray<2, Float>   // 8 bytes, 스택에
}

// ❌ 주의: 너무 큰 InlineArray는 스택 오버플로우 위험
// let huge: InlineArray<1_000_000, Int>  // 스택에 8MB!
```

---

## 관련 문서

- [SE-0453: Vector (InlineArray)](https://github.com/swiftlang/swift-evolution/blob/main/proposals/0453-vector.md)
- [SE-0447: Span](https://github.com/swiftlang/swift-evolution/blob/main/proposals/0447-span-access-shared-contiguous-storage.md)
- [What's new in Swift (250)](https://developer.apple.com/videos/play/wwdc2025/250/)
