# SwiftData Class Inheritance

> WWDC25 | iOS 26, macOS Tahoe  
> 📖 https://developer.apple.com/documentation/SwiftData/Adopting-inheritance-in-SwiftData  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/291/

---

## 개요

iOS 26에서 SwiftData가 **클래스 상속(Class Inheritance)**을 지원합니다. @Model 클래스 간의 상속 계층을 정의하여, 공통 속성은 부모에 두고 특화된 속성은 자식에 둘 수 있습니다.

---

## 기본 사용법

### 상속 계층 정의

```swift
import SwiftData

// 기본 모델 (부모)
@Model
class Animal {
    var name: String
    var age: Int
    var weight: Double
    
    init(name: String, age: Int, weight: Double) {
        self.name = name
        self.age = age
        self.weight = weight
    }
}

// 자식 모델
@Model
class Dog: Animal {
    var breed: String
    var isGoodBoy: Bool
    
    init(name: String, age: Int, weight: Double, breed: String) {
        self.breed = breed
        self.isGoodBoy = true
        super.init(name: name, age: age, weight: weight)
    }
}

@Model
class Cat: Animal {
    var isIndoor: Bool
    var favoriteSpot: String?
    
    init(name: String, age: Int, weight: Double, isIndoor: Bool) {
        self.isIndoor = isIndoor
        super.init(name: name, age: age, weight: weight)
    }
}
```

---

## ModelContainer 설정

```swift
import SwiftData
import SwiftUI

@main
struct MyApp: App {
    var body: some Scene {
        WindowGroup {
            ContentView()
        }
        .modelContainer(for: [
            Animal.self,  // 부모 모델 등록 → 자식도 자동 포함
        ])
    }
}
```

> ⚠️ 부모 모델을 등록하면 모든 자식 모델이 자동으로 포함됩니다.

---

## 쿼리

### 부모 타입으로 모든 자식 조회

```swift
struct AnimalListView: View {
    // Animal로 쿼리하면 Dog, Cat 등 모든 하위 타입 포함
    @Query var animals: [Animal]
    
    var body: some View {
        List(animals) { animal in
            HStack {
                // 타입에 따른 아이콘 표시
                if animal is Dog {
                    Image(systemName: "dog.fill")
                } else if animal is Cat {
                    Image(systemName: "cat.fill")
                }
                
                VStack(alignment: .leading) {
                    Text(animal.name)
                    Text("나이: \(animal.age)세")
                        .font(.caption)
                }
            }
        }
    }
}
```

### 특정 자식 타입만 조회

```swift
struct DogListView: View {
    // Dog 타입만 조회
    @Query var dogs: [Dog]
    
    var body: some View {
        List(dogs) { dog in
            VStack(alignment: .leading) {
                Text(dog.name)
                Text("품종: \(dog.breed)")
                    .font(.caption)
            }
        }
    }
}
```

### 다운캐스팅으로 타입별 처리

```swift
func processAnimal(_ animal: Animal) {
    // 기본 속성 접근
    print("이름: \(animal.name), 나이: \(animal.age)")
    
    // 타입별 특화 처리
    switch animal {
    case let dog as Dog:
        print("품종: \(dog.breed)")
    case let cat as Cat:
        print("실내 고양이: \(cat.isIndoor)")
    default:
        print("일반 동물")
    }
}
```

---

## 관계 (Relationships)

```swift
@Model
class Owner {
    var name: String
    
    // 부모 타입으로 관계 설정 → 모든 자식 타입 저장 가능
    @Relationship(deleteRule: .cascade)
    var pets: [Animal]
    
    init(name: String, pets: [Animal] = []) {
        self.name = name
        self.pets = pets
    }
}

// 사용
let owner = Owner(name: "김철수")
owner.pets.append(Dog(name: "멍멍이", age: 3, weight: 12.5, breed: "시바견"))
owner.pets.append(Cat(name: "나비", age: 2, weight: 4.0, isIndoor: true))
```

---

## 마이그레이션

```swift
// 상속 계층 변경 시 마이그레이션 필요
enum AnimalSchemaV1: VersionedSchema {
    static var versionIdentifier = Schema.Version(1, 0, 0)
    static var models: [any PersistentModel.Type] = [Animal.self]
    
    @Model
    class Animal {
        var name: String
        var age: Int
    }
}

enum AnimalSchemaV2: VersionedSchema {
    static var versionIdentifier = Schema.Version(2, 0, 0)
    static var models: [any PersistentModel.Type] = [Animal.self, Dog.self, Cat.self]
    
    @Model
    class Animal {
        var name: String
        var age: Int
        var weight: Double
    }
    
    @Model
    class Dog: Animal {
        var breed: String
    }
    
    @Model
    class Cat: Animal {
        var isIndoor: Bool
    }
}
```

---

## 모범 사례

1. **얕은 상속 계층**: 2-3 레벨까지 권장
2. **공통 속성은 부모에**: 중복 방지
3. **쿼리 성능**: 특정 타입만 필요하면 자식 타입으로 직접 쿼리
4. **프로토콜 활용 고려**: 단순 속성 공유는 프로토콜이 더 적합할 수 있음
5. **관계는 부모 타입으로**: 다형성 활용

---

## 관련 세션

- [Model your schema with SwiftData (291)](https://developer.apple.com/videos/play/wwdc2025/291/)
