# SwiftUI AlarmKit Integration

> WWDC25 | iOS 26, watchOS 26  
> 📖 https://developer.apple.com/documentation/AlarmKit

---

## 개요

**AlarmKit**은 iOS 26에서 새로 도입된 프레임워크로, 앱에서 **눈에 띄는 알람과 카운트다운**을 예약할 수 있습니다. 시스템 알람 앱과 동일한 수준의 중요도로 사용자에게 알림을 전달합니다.

---

## 기본 사용법

### 알람 예약

```swift
import AlarmKit

// 특정 시간에 알람 예약
let alarm = Alarm(
    id: "morning-alarm",
    title: "아침 기상",
    scheduledTime: DateComponents(hour: 7, minute: 0),
    sound: .default
)

try await AlarmManager.shared.schedule(alarm)
```

### 카운트다운 타이머

```swift
// 카운트다운 예약
let countdown = Countdown(
    id: "cooking-timer",
    title: "라면 끓이기",
    duration: .minutes(3),
    sound: .alarm
)

try await AlarmManager.shared.schedule(countdown)
```

---

## SwiftUI 통합

### 알람 뷰

```swift
import SwiftUI
import AlarmKit

struct AlarmSetupView: View {
    @State private var wakeUpTime = Date()
    @State private var isAlarmSet = false
    
    var body: some View {
        VStack(spacing: 20) {
            DatePicker(
                "기상 시간",
                selection: $wakeUpTime,
                displayedComponents: .hourAndMinute
            )
            .datePickerStyle(.wheel)
            
            Button(isAlarmSet ? "알람 해제" : "알람 설정") {
                Task {
                    if isAlarmSet {
                        try await cancelAlarm()
                    } else {
                        try await setAlarm()
                    }
                    isAlarmSet.toggle()
                }
            }
            .buttonStyle(.borderedProminent)
            .controlSize(.large)
        }
        .padding()
    }
    
    func setAlarm() async throws {
        let components = Calendar.current.dateComponents(
            [.hour, .minute], from: wakeUpTime
        )
        
        let alarm = Alarm(
            id: "wake-up",
            title: "기상 알람",
            scheduledTime: components,
            sound: .default
        )
        
        try await AlarmManager.shared.schedule(alarm)
    }
    
    func cancelAlarm() async throws {
        try await AlarmManager.shared.cancel(id: "wake-up")
    }
}
```

### 카운트다운 뷰

```swift
struct CountdownView: View {
    @State private var minutes: Int = 5
    @State private var activeCountdown: Countdown?
    
    var body: some View {
        VStack(spacing: 24) {
            // 시간 선택
            Picker("분", selection: $minutes) {
                ForEach(1...60, id: \.self) { min in
                    Text("\(min)분").tag(min)
                }
            }
            .pickerStyle(.wheel)
            
            // 카운트다운 시작/정지
            if let countdown = activeCountdown {
                // 활성 카운트다운 표시
                Text("남은 시간")
                    .font(.headline)
                
                // 시스템 카운트다운 라이브 디스플레이
                CountdownDisplayView(countdown: countdown)
                
                Button("취소", role: .destructive) {
                    Task {
                        try await AlarmManager.shared.cancel(id: countdown.id)
                        activeCountdown = nil
                    }
                }
            } else {
                Button("시작") {
                    Task {
                        let cd = Countdown(
                            id: UUID().uuidString,
                            title: "\(minutes)분 타이머",
                            duration: .minutes(minutes),
                            sound: .alarm
                        )
                        try await AlarmManager.shared.schedule(cd)
                        activeCountdown = cd
                    }
                }
                .buttonStyle(.borderedProminent)
                .controlSize(.large)
            }
        }
        .padding()
    }
}
```

---

## 알람 관리

### 예약된 알람 조회

```swift
// 모든 예약된 알람/카운트다운 조회
let scheduled = try await AlarmManager.shared.scheduledAlarms()

for alarm in scheduled {
    print("ID: \(alarm.id)")
    print("제목: \(alarm.title)")
    print("시간: \(alarm.scheduledTime)")
}
```

### 알람 취소

```swift
// 특정 알람 취소
try await AlarmManager.shared.cancel(id: "morning-alarm")

// 모든 알람 취소
try await AlarmManager.shared.cancelAll()
```

---

## 사운드 옵션

```swift
// 기본 사운드
let alarm1 = Alarm(id: "a1", title: "알람", 
                   scheduledTime: components, 
                   sound: .default)

// 시스템 알람 사운드
let alarm2 = Alarm(id: "a2", title: "알람", 
                   scheduledTime: components, 
                   sound: .alarm)

// 커스텀 사운드 (번들 리소스)
let alarm3 = Alarm(id: "a3", title: "알람", 
                   scheduledTime: components, 
                   sound: .custom(named: "my-alarm-sound"))
```

---

## 권한 및 제한 사항

- 사용자 권한 요청 필요 (알림 권한과 별도)
- 알람은 **방해 금지 모드에서도 울림** (시스템 알람과 동일)
- 카운트다운은 Live Activity로 표시될 수 있음
- 앱당 예약 가능한 알람 수 제한 있음

```swift
// 권한 요청
let status = try await AlarmManager.shared.requestAuthorization()

switch status {
case .authorized:
    print("알람 권한 허용됨")
case .denied:
    print("알람 권한 거부됨")
case .notDetermined:
    print("아직 결정되지 않음")
}
```

---

## 관련 문서

- [AlarmKit 공식 문서](https://developer.apple.com/documentation/AlarmKit)
