# SwiftUI: Implementing Liquid Glass Design

> WWDC25 | iOS 26, macOS Tahoe, visionOS 26  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/323/  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/269/  
> 📖 https://developer.apple.com/documentation/SwiftUI/Applying-Liquid-Glass-to-custom-views

---

## 개요

iOS 26의 가장 큰 시각적 변화인 **Liquid Glass**는 반투명 유리 머티리얼을 시스템 전체에 적용합니다. SwiftUI에서는 기존 코드에 자동 적용되며, 커스텀 뷰에도 새로운 API를 제공합니다.

---

## 자동 적용 (기존 코드)

다음 컴포넌트는 iOS 26에서 자동으로 Liquid Glass가 적용됩니다:

- **NavigationBar** → Glass 머티리얼 배경
- **TabView** → Glass 탭 바
- **Toolbar** → Glass 그룹핑
- **Sheet/Popover** → Glass 배경
- **Toggle, Slider, Picker** → Glass 스타일

```swift
// 기존 코드 변경 없이 자동 적용
NavigationStack {
    List(items) { item in
        Text(item.name)
    }
    .navigationTitle("목록")
    .toolbar {
        Button("추가", systemImage: "plus") { }
    }
}
```

---

## Glass 컨테이너 효과

### .glassEffect() 모디파이어

```swift
// 커스텀 뷰에 Glass 효과 적용
struct GlassCardView: View {
    var body: some View {
        VStack {
            Image(systemName: "star.fill")
                .font(.largeTitle)
            Text("즐겨찾기")
        }
        .padding()
        .glassEffect()  // Glass 머티리얼 적용
    }
}
```

### Glass 효과 변형

```swift
// 기본 Glass
.glassEffect()

// 인터랙티브 Glass (버튼 등)
.glassEffect(.interactive)

// 레귤러 Glass (정보 표시용)
.glassEffect(.regular)

// 틴트 적용
.glassEffect()
.tint(.blue)
```

---

## 탭 뷰 커스터마이징

### 기본 탭 뷰 (자동 Glass)

```swift
TabView {
    Tab("홈", systemImage: "house.fill") {
        HomeView()
    }
    Tab("검색", systemImage: "magnifyingglass") {
        SearchView()
    }
    Tab("프로필", systemImage: "person.fill") {
        ProfileView()
    }
}
```

### Floating Tab Bar (신규)

```swift
TabView {
    Tab("홈", systemImage: "house.fill") {
        HomeView()
    }
    Tab("검색", systemImage: "magnifyingglass") {
        SearchView()
    }
}
.tabViewStyle(.floatingTabBar)  // 떠 있는 Glass 탭 바
```

---

## 툴바 Glass 그룹핑

```swift
.toolbar {
    // 같은 placement의 아이템은 하나의 Glass에 그룹핑
    ToolbarItemGroup(placement: .primaryAction) {
        Button("편집", systemImage: "pencil") { }
        Button("공유", systemImage: "square.and.arrow.up") { }
    }
    
    // 별도의 Glass 그룹
    ToolbarItem(placement: .secondaryAction) {
        Button("설정", systemImage: "gear") { }
    }
}
```

---

## Scroll Edge 효과

콘텐츠가 Glass 영역 아래로 스크롤될 때의 전환 효과:

```swift
ScrollView {
    LazyVStack {
        ForEach(items) { item in
            ItemRow(item: item)
        }
    }
}
// Soft edge: 점진적 페이드와 블러 (기본값)
.scrollEdgeEffect(.soft)

// Hard edge: 불투명 배경
.scrollEdgeEffect(.hard)
```

---

## 커스텀 Glass 뷰

### 플로팅 액션 버튼

```swift
struct FloatingActionButton: View {
    var body: some View {
        Button(action: { }) {
            Image(systemName: "plus")
                .font(.title2)
                .foregroundStyle(.primary)
        }
        .frame(width: 56, height: 56)
        .glassEffect(.interactive)
        .clipShape(Circle())
    }
}
```

### Glass 카드 그리드

```swift
struct GlassCardGrid: View {
    let items: [CardItem]
    
    var body: some View {
        LazyVGrid(columns: [
            GridItem(.flexible()),
            GridItem(.flexible())
        ], spacing: 16) {
            ForEach(items) { item in
                VStack(spacing: 8) {
                    Image(systemName: item.icon)
                        .font(.title)
                    Text(item.title)
                        .font(.caption)
                }
                .frame(height: 120)
                .frame(maxWidth: .infinity)
                .glassEffect()
            }
        }
        .padding()
    }
}
```

---

## 디자인 가이드라인

### ✅ 올바른 사용

- 플로팅 UI 요소 (FAB, 오버레이 컨트롤)
- 정보 카드와 패널
- 인터랙티브 컨트롤 그룹
- 네비게이션과 탭 바

### ❌ 피해야 할 사용

- **Glass on Glass**: Glass 위에 Glass 중첩 금지
- **모든 곳에 Glass**: 과도한 사용은 가독성 저하
- **작은 텍스트 위의 Glass**: 읽기 어려울 수 있음
- **브랜드 색상으로 Glass 틴팅**: Glass 자체가 아닌 콘텐츠에 브랜드 색상 사용

---

## 조건부 Glass (레거시 지원)

```swift
// iOS 26 이상에서만 Glass 효과 적용
var body: some View {
    MyView()
        .modifier(GlassModifier())
}

struct GlassModifier: ViewModifier {
    func body(content: Content) -> some View {
        if #available(iOS 26, *) {
            content.glassEffect()
        } else {
            content
                .background(.ultraThinMaterial)
                .clipShape(RoundedRectangle(cornerRadius: 16))
        }
    }
}
```

---

## 관련 세션

- [Build a SwiftUI app with the new design (323)](https://developer.apple.com/videos/play/wwdc2025/323/)
- [Meet Liquid Glass (269)](https://developer.apple.com/videos/play/wwdc2025/269/)
- [Get to know the new design system (252)](https://developer.apple.com/videos/play/wwdc2025/252/)
