# SwiftUI Styled Text Editing

> WWDC25 | iOS 26, macOS Tahoe  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/258/  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/256/

---

## 개요

iOS 26에서 **TextView**가 도입되어, SwiftUI에서 **스타일이 적용된 리치 텍스트 편집**이 가능해졌습니다. AttributedString 바인딩으로 볼드, 이탤릭, 링크 등 서식을 지원합니다.

---

## TextView (신규)

### 기본 사용

```swift
import SwiftUI

struct RichEditorView: View {
    @State private var text: AttributedString = "여기에 입력하세요"
    
    var body: some View {
        // TextView - 리치 텍스트 편집 전용 뷰
        TextView(text: $text)
            .padding()
    }
}
```

### 초기 스타일 지정

```swift
struct StyledEditorView: View {
    @State private var text: AttributedString = {
        var str = AttributedString("")
        
        // 제목
        var title = AttributedString("프로젝트 노트\n")
        title.font = .title.bold()
        
        // 본문
        var body = AttributedString("여기에 내용을 작성하세요...")
        body.font = .body
        body.foregroundColor = .secondary
        
        str.append(title)
        str.append(body)
        return str
    }()
    
    var body: some View {
        TextView(text: $text)
            .frame(minHeight: 300)
            .padding()
    }
}
```

---

## 서식 도구 연동

### 서식 툴바

```swift
struct FormattedEditorView: View {
    @State private var text: AttributedString = "편집할 텍스트"
    @State private var selection: Range<AttributedString.Index>?
    
    var body: some View {
        VStack {
            // 서식 툴바
            HStack(spacing: 16) {
                Button("B") { applyBold() }
                    .font(.body.bold())
                
                Button("I") { applyItalic() }
                    .font(.body.italic())
                
                Button("U") { applyUnderline() }
                    .underline()
                
                Divider()
                    .frame(height: 20)
                
                // 색상 선택
                ColorPicker("", selection: $textColor)
                    .labelsHidden()
            }
            .padding(.horizontal)
            
            // 편집기
            TextView(text: $text)
                .padding()
        }
    }
    
    @State private var textColor: Color = .primary
    
    func applyBold() {
        guard let selection = selection else { return }
        text[selection].font = .body.bold()
    }
    
    func applyItalic() {
        guard let selection = selection else { return }
        text[selection].font = .body.italic()
    }
    
    func applyUnderline() {
        guard let selection = selection else { return }
        text[selection].underlineStyle = .single
    }
}
```

---

## TextEditor 업데이트

기존 TextEditor도 AttributedString 바인딩을 지원합니다.

```swift
struct BasicRichEditor: View {
    @State private var text: AttributedString = "텍스트 편집기"
    
    var body: some View {
        TextEditor(text: $text)  // iOS 26: AttributedString 바인딩
            .frame(minHeight: 200)
    }
}
```

---

## Markdown → 리치 텍스트

```swift
struct MarkdownEditorView: View {
    @State private var richText: AttributedString
    
    init() {
        let markdown = """
        # 제목
        
        **볼드 텍스트**와 *이탤릭 텍스트*
        
        - 항목 1
        - 항목 2
        
        [링크](https://developer.apple.com)
        """
        
        _richText = State(
            initialValue: (try? AttributedString(markdown: markdown)) 
                ?? AttributedString(markdown)
        )
    }
    
    var body: some View {
        TextView(text: $richText)
            .padding()
    }
}
```

---

## 스타일 속성

### 지원 속성

```swift
var styled = AttributedString("스타일 텍스트")

// 폰트
styled.font = .system(size: 18, weight: .semibold, design: .rounded)

// 전경색
styled.foregroundColor = .blue

// 배경색
styled.backgroundColor = .yellow.opacity(0.3)

// 밑줄
styled.underlineStyle = .single
styled.underlineColor = .red

// 취소선
styled.strikethroughStyle = .single

// 자간
styled.kern = 2.0

// 기준선 오프셋
styled.baselineOffset = 5.0

// 링크
styled.link = URL(string: "https://apple.com")
```

---

## 읽기 전용 스타일 텍스트

편집 없이 리치 텍스트만 표시할 때는 `Text`를 사용합니다.

```swift
struct RichDisplayView: View {
    var body: some View {
        Text(attributedContent)
            .textSelection(.enabled)  // 선택만 허용
    }
    
    var attributedContent: AttributedString {
        var result = AttributedString()
        
        var title = AttributedString("공지사항\n\n")
        title.font = .title2.bold()
        result.append(title)
        
        var body = AttributedString("새로운 기능이 추가되었습니다. ")
        body.font = .body
        result.append(body)
        
        var link = AttributedString("자세히 보기")
        link.foregroundColor = .blue
        link.link = URL(string: "https://example.com")
        result.append(link)
        
        return result
    }
}
```

---

## 관련 세션

- [Code-along: Cook up a rich text experience (258)](https://developer.apple.com/videos/play/wwdc2025/258/)
- [What's new in SwiftUI (256)](https://developer.apple.com/videos/play/wwdc2025/256/)
