# SwiftUI WebKit Integration

> WWDC25 | iOS 26, macOS Tahoe  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/297/

---

## 개요

iOS 26에서 **WebView**가 SwiftUI 네이티브 뷰로 도입되었습니다. WKWebView를 UIViewRepresentable로 감쌀 필요 없이, SwiftUI 선언적 문법으로 웹 콘텐츠를 직접 표시하고 제어할 수 있습니다.

---

## 기본 사용법

### URL 로드

```swift
import SwiftUI
import WebKit

struct SimpleWebView: View {
    var body: some View {
        WebView(url: URL(string: "https://developer.apple.com")!)
    }
}
```

### URL 바인딩

```swift
struct BrowsingView: View {
    @State private var url = URL(string: "https://apple.com")!
    
    var body: some View {
        VStack {
            // URL 입력 바
            TextField("URL", text: Binding(
                get: { url.absoluteString },
                set: { url = URL(string: $0) ?? url }
            ))
            .textFieldStyle(.roundedBorder)
            .autocapitalization(.none)
            .padding(.horizontal)
            
            // 웹 뷰
            WebView(url: url)
        }
    }
}
```

---

## 네비게이션 제어

### WebViewState

```swift
struct NavigableWebView: View {
    @State private var webState = WebViewState()
    
    var body: some View {
        VStack {
            // 네비게이션 바
            HStack {
                Button(action: { webState.goBack() }) {
                    Image(systemName: "chevron.left")
                }
                .disabled(!webState.canGoBack)
                
                Button(action: { webState.goForward() }) {
                    Image(systemName: "chevron.right")
                }
                .disabled(!webState.canGoForward)
                
                Spacer()
                
                if webState.isLoading {
                    ProgressView()
                }
                
                Button(action: { webState.reload() }) {
                    Image(systemName: "arrow.clockwise")
                }
            }
            .padding(.horizontal)
            
            // 로딩 프로그레스
            if webState.isLoading {
                ProgressView(value: webState.estimatedProgress)
            }
            
            // 페이지 타이틀
            Text(webState.title ?? "")
                .font(.caption)
                .foregroundStyle(.secondary)
            
            // 웹 뷰
            WebView(url: URL(string: "https://apple.com")!, state: $webState)
        }
    }
}
```

---

## 프로퍼티 접근

```swift
struct WebDetailView: View {
    @State private var webState = WebViewState()
    
    var body: some View {
        VStack(alignment: .leading) {
            // 현재 페이지 정보
            Group {
                Text("제목: \(webState.title ?? "로딩 중...")")
                Text("URL: \(webState.currentURL?.absoluteString ?? "-")")
                Text("로딩: \(webState.isLoading ? "예" : "아니오")")
                Text("진행률: \(Int(webState.estimatedProgress * 100))%")
            }
            .font(.caption)
            .padding(.horizontal)
            
            WebView(url: startURL, state: $webState)
        }
    }
    
    let startURL = URL(string: "https://developer.apple.com")!
}
```

---

## JavaScript 실행

```swift
struct InteractiveWebView: View {
    @State private var webState = WebViewState()
    @State private var result: String = ""
    
    var body: some View {
        VStack {
            WebView(url: url, state: $webState)
            
            Button("페이지 제목 가져오기") {
                Task {
                    let title = try await webState.evaluateJavaScript(
                        "document.title"
                    )
                    result = title as? String ?? "알 수 없음"
                }
            }
            
            Text("결과: \(result)")
                .font(.caption)
        }
    }
    
    let url = URL(string: "https://apple.com")!
}
```

---

## HTML 문자열 로드

```swift
struct LocalHTMLView: View {
    var body: some View {
        WebView(html: """
        <!DOCTYPE html>
        <html>
        <head>
            <meta name="viewport" content="width=device-width, initial-scale=1">
            <style>
                body {
                    font-family: -apple-system;
                    padding: 20px;
                    background: transparent;
                }
                h1 { color: #333; }
                p { color: #666; line-height: 1.6; }
            </style>
        </head>
        <body>
            <h1>SwiftUI WebView</h1>
            <p>HTML 콘텐츠를 직접 렌더링합니다.</p>
        </body>
        </html>
        """)
    }
}
```

---

## 로컬 파일 로드

```swift
struct LocalFileWebView: View {
    var body: some View {
        // 번들 내 HTML 파일
        if let fileURL = Bundle.main.url(
            forResource: "guide", 
            withExtension: "html"
        ) {
            WebView(url: fileURL)
        }
    }
}
```

---

## 네비게이션 정책

```swift
struct PolicyWebView: View {
    var body: some View {
        WebView(url: URL(string: "https://apple.com")!)
            .webViewNavigationPolicy { request in
                // 특정 도메인만 허용
                guard let host = request.url?.host else {
                    return .cancel
                }
                
                if host.contains("apple.com") {
                    return .allow
                } else {
                    // 외부 링크는 Safari에서 열기
                    return .cancel
                }
            }
    }
}
```

---

## WKWebView 대비 장점

| 기능 | WKWebView (UIKit) | WebView (SwiftUI) |
|------|-------------------|-------------------|
| 선언적 문법 | ❌ | ✅ |
| 상태 바인딩 | 수동 KVO | @State 바인딩 |
| 네비게이션 | Delegate 패턴 | 클로저/모디파이어 |
| Wrapper 필요 | UIViewRepresentable | 불필요 |
| 환경 값 접근 | 불가 | @Environment |

---

## 관련 세션

- [Build a browser with WebKit and SwiftUI (297)](https://developer.apple.com/videos/play/wwdc2025/297/)
