# UIKit: Implementing Liquid Glass Design

> WWDC25 | iOS 26  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/275/  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/269/

---

## 개요

iOS 26에서 UIKit도 **Liquid Glass** 디자인을 전면 도입합니다. UINavigationBar, UITabBar, UIToolbar 등 시스템 UI가 자동으로 Glass 머티리얼로 업데이트되며, 커스텀 뷰에도 Glass 효과를 적용할 수 있습니다.

---

## 자동 적용되는 변경 사항

### UINavigationBar

- Glass 머티리얼 배경 자동 적용
- 바 버튼 아이템이 Glass 캡슐로 그룹핑
- 스크롤 시 Scroll Edge Effect 자동 적용

```swift
// 기존 코드 변경 없이 자동으로 Glass 적용
let navController = UINavigationController(rootViewController: viewController)

// 네비게이션 바 아이템 - 자동 Glass 그룹핑
navigationItem.rightBarButtonItems = [
    UIBarButtonItem(systemItem: .edit),
    UIBarButtonItem(systemItem: .action)
]
```

### UITabBar

```swift
// 탭 바도 자동으로 Glass 적용
let tabBarController = UITabBarController()
tabBarController.viewControllers = [vc1, vc2, vc3]
// → Glass 머티리얼 탭 바로 자동 변환
```

---

## UIGlassEffect API

### 뷰에 Glass 효과 적용

```swift
import UIKit

class GlassCardViewController: UIViewController {
    override func viewDidLoad() {
        super.viewDidLoad()
        
        let cardView = UIView()
        cardView.frame = CGRect(x: 20, y: 100, width: 340, height: 200)
        
        // Glass 효과 적용
        let glassEffect = UIGlassEffect()
        let glassView = UIVisualEffectView(effect: glassEffect)
        glassView.frame = cardView.bounds
        glassView.autoresizingMask = [.flexibleWidth, .flexibleHeight]
        
        cardView.addSubview(glassView)
        
        // 콘텐츠를 contentView에 추가
        let label = UILabel()
        label.text = "Glass 카드"
        label.font = .preferredFont(forTextStyle: .headline)
        label.translatesAutoresizingMaskIntoConstraints = false
        glassView.contentView.addSubview(label)
        
        NSLayoutConstraint.activate([
            label.centerXAnchor.constraint(equalTo: glassView.contentView.centerXAnchor),
            label.centerYAnchor.constraint(equalTo: glassView.contentView.centerYAnchor)
        ])
        
        view.addSubview(cardView)
    }
}
```

### 인터랙티브 Glass

```swift
// 버튼에 Glass 효과
class GlassButtonViewController: UIViewController {
    override func viewDidLoad() {
        super.viewDidLoad()
        
        var config = UIButton.Configuration.glass()  // 신규
        config.title = "Glass 버튼"
        config.image = UIImage(systemName: "star.fill")
        
        let button = UIButton(configuration: config)
        button.addTarget(self, action: #selector(tapped), for: .touchUpInside)
        
        view.addSubview(button)
    }
    
    @objc func tapped() { }
}
```

---

## Scroll Edge Effect

### 네비게이션 바 스크롤 엣지

```swift
class ScrollEdgeViewController: UIViewController {
    override func viewDidLoad() {
        super.viewDidLoad()
        
        let tableView = UITableView(frame: view.bounds, style: .plain)
        view.addSubview(tableView)
        
        // Scroll Edge 효과는 UIScrollView에 자동 적용
        // 커스터마이징:
        let appearance = UINavigationBarAppearance()
        appearance.configureWithDefaultBackground()  // Glass 배경
        
        // 스크롤 엣지 설정
        navigationController?.navigationBar.scrollEdgeAppearance = appearance
    }
}
```

---

## UIBarButtonItem 그룹핑

```swift
// iOS 26: 바 버튼 아이템의 Glass 그룹핑 제어
class ToolbarViewController: UIViewController {
    override func viewDidLoad() {
        super.viewDidLoad()
        
        // 그룹 1: 편집 도구
        let editGroup = UIBarButtonItemGroup(
            barButtonItems: [
                UIBarButtonItem(systemItem: .edit),
                UIBarButtonItem(image: UIImage(systemName: "scissors"))
            ],
            representativeItem: nil
        )
        
        // 그룹 2: 공유
        let shareGroup = UIBarButtonItemGroup(
            barButtonItems: [
                UIBarButtonItem(systemItem: .action)
            ],
            representativeItem: nil
        )
        
        // 그룹별로 별도의 Glass 캡슐에 표시
        navigationItem.trailingItemGroups = [editGroup, shareGroup]
    }
}
```

---

## 틴트 색상

```swift
// Glass에 틴트 색상 적용
let glassEffect = UIGlassEffect()
glassEffect.tintColor = .systemBlue

let glassView = UIVisualEffectView(effect: glassEffect)
```

---

## UIVisualEffectView 업데이트

```swift
// 기존 블러 효과와의 비교
// iOS 25 이전
let blur = UIBlurEffect(style: .systemThinMaterial)
let blurView = UIVisualEffectView(effect: blur)

// iOS 26: Glass 효과
let glass = UIGlassEffect()
let glassView = UIVisualEffectView(effect: glass)

// 조건부 사용
if #available(iOS 26, *) {
    let effect = UIGlassEffect()
    effectView = UIVisualEffectView(effect: effect)
} else {
    let effect = UIBlurEffect(style: .systemThinMaterial)
    effectView = UIVisualEffectView(effect: effect)
}
```

---

## 옵트아웃

```xml
<!-- Info.plist에서 임시로 Liquid Glass 비활성화 -->
<key>UIDesignRequiresLiquidGlass</key>
<false/>
```

> ⚠️ 이 키는 임시용이며 추후 릴리스에서 제거 예정입니다.

---

## 마이그레이션 체크리스트

- [ ] 커스텀 네비게이션 바 배경이 Glass와 충돌하지 않는지 확인
- [ ] Glass on Glass 중첩이 없는지 확인
- [ ] 불투명 배경 뷰가 Glass 뒤에서 올바르게 보이는지 확인
- [ ] 스크롤 엣지 효과가 콘텐츠와 잘 어울리는지 확인
- [ ] 다크 모드에서 Glass가 올바르게 전환되는지 확인
- [ ] 접근성: Glass 위 텍스트 대비가 충분한지 확인

---

## 관련 세션

- [Build a UIKit app with the new design (275)](https://developer.apple.com/videos/play/wwdc2025/275/)
- [Meet Liquid Glass (269)](https://developer.apple.com/videos/play/wwdc2025/269/)
- [Get to know the new design system (252)](https://developer.apple.com/videos/play/wwdc2025/252/)
