# WidgetKit: Implementing Liquid Glass Design

> WWDC25 | iOS 26, watchOS 26  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/232/

---

## 개요

iOS 26에서 위젯이 **Liquid Glass** 디자인으로 업데이트됩니다. 위젯 배경이 Glass 머티리얼로 변경되고, Live Activities가 CarPlay에서도 지원됩니다.

---

## 위젯 Glass 자동 적용

### 기본 위젯

기존 위젯은 iOS 26에서 자동으로 Glass 배경이 적용됩니다.

```swift
import WidgetKit
import SwiftUI

struct MyWidget: Widget {
    let kind: String = "MyWidget"
    
    var body: some WidgetConfiguration {
        StaticConfiguration(kind: kind, provider: Provider()) { entry in
            MyWidgetEntryView(entry: entry)
                // iOS 26: 자동으로 Glass 배경 적용
                .containerBackground(.fill.tertiary, for: .widget)
        }
        .configurationDisplayName("내 위젯")
        .description("앱 정보를 한눈에")
        .supportedFamilies([.systemSmall, .systemMedium, .systemLarge])
    }
}
```

### Glass 배경 커스터마이징

```swift
struct MyWidgetEntryView: View {
    var entry: Provider.Entry
    
    var body: some View {
        VStack(alignment: .leading) {
            HStack {
                Image(systemName: "chart.line.uptrend.xyaxis")
                    .foregroundStyle(.blue)
                Text("오늘의 통계")
                    .font(.headline)
            }
            
            Text("\(entry.count)")
                .font(.system(size: 48, weight: .bold, design: .rounded))
            
            Text("전일 대비 +12%")
                .font(.caption)
                .foregroundStyle(.secondary)
        }
        .containerBackground(for: .widget) {
            // 커스텀 Glass 배경
            Color.clear  // Glass가 자동 적용되므로 투명 배경 사용
        }
    }
}
```

---

## Glass 효과가 적용된 위젯 요소

### 인터랙티브 위젯

```swift
struct InteractiveWidgetView: View {
    var entry: Provider.Entry
    
    var body: some View {
        VStack {
            Text("할 일")
                .font(.headline)
            
            ForEach(entry.tasks.prefix(3)) { task in
                // 토글 버튼 - Glass 인터랙티브 스타일 자동 적용
                Button(intent: ToggleTaskIntent(id: task.id)) {
                    HStack {
                        Image(systemName: task.isComplete 
                            ? "checkmark.circle.fill" 
                            : "circle")
                        Text(task.title)
                            .strikethrough(task.isComplete)
                    }
                }
                .buttonStyle(.plain)
            }
        }
    }
}
```

---

## Live Activities on CarPlay (신규)

### CarPlay Live Activity 지원

```swift
import ActivityKit

struct DeliveryAttributes: ActivityAttributes {
    public struct ContentState: Codable, Hashable {
        var status: String
        var estimatedArrival: Date
        var driverName: String
    }
    
    var orderNumber: String
}

// Live Activity 시작
let attributes = DeliveryAttributes(orderNumber: "ORD-12345")
let state = DeliveryAttributes.ContentState(
    status: "배달 중",
    estimatedArrival: Date().addingTimeInterval(1800),
    driverName: "김운전"
)

let activity = try Activity.request(
    attributes: attributes,
    content: .init(state: state, staleDate: nil)
)
```

### CarPlay용 UI 정의

```swift
struct DeliveryLiveActivity: Widget {
    var body: some WidgetConfiguration {
        ActivityConfiguration(for: DeliveryAttributes.self) { context in
            // Lock Screen / Dynamic Island
            DeliveryLockScreenView(context: context)
        } dynamicIsland: { context in
            DynamicIsland {
                DynamicIslandExpandedRegion(.leading) {
                    Image(systemName: "car.fill")
                }
                DynamicIslandExpandedRegion(.trailing) {
                    Text(context.state.estimatedArrival, style: .timer)
                }
                DynamicIslandExpandedRegion(.bottom) {
                    Text(context.state.status)
                }
            } compactLeading: {
                Image(systemName: "car.fill")
            } compactTrailing: {
                Text(context.state.estimatedArrival, style: .timer)
            } minimal: {
                Image(systemName: "car.fill")
            }
        }
        // CarPlay에서도 Live Activity 표시 (iOS 26)
    }
}
```

---

## Control Center 위젯

```swift
struct QuickToggleControl: ControlWidget {
    static let kind = "com.myapp.quickToggle"
    
    var body: some ControlWidgetConfiguration {
        StaticControlConfiguration(kind: Self.kind) {
            ControlWidgetToggle(
                "자동 모드",
                isOn: AutoModeBinding(),
                action: ToggleAutoModeIntent()
            ) { isOn in
                Label(
                    isOn ? "켜짐" : "꺼짐",
                    systemImage: isOn ? "auto.fill" : "auto"
                )
            }
        }
        .displayName("자동 모드")
        .description("자동 모드를 켜고 끕니다")
    }
}
```

---

## Glass 디자인 가이드라인 (위젯)

### ✅ 올바른 사용

- **시스템 폰트 사용**: Dynamic Type 자동 지원
- **SF Symbols**: Glass 위에서 선명하게 보임
- **충분한 여백**: Glass 효과를 위한 패딩
- **간결한 정보**: 핵심 데이터만 표시

### ❌ 피해야 할 것

- **과도한 배경색**: Glass의 투명 효과를 가리지 않기
- **작은 텍스트**: Glass 위에서 읽기 어려움
- **복잡한 레이아웃**: 위젯은 간결함이 핵심

---

## 관련 세션

- [Design for widgets, Live Activities, and controls (232)](https://developer.apple.com/videos/play/wwdc2025/232/)
