# Widgets for visionOS

> WWDC25 | visionOS 26  
> 🎬 https://developer.apple.com/videos/play/wwdc2025/267/

---

## 개요

visionOS 26에서 **위젯이 공간 컴퓨팅 환경에 도입**되었습니다. 사용자의 공간에 떠 있는 Glass 카드 형태로 앱 정보를 표시합니다.

---

## visionOS 위젯 기본

### 지원 위젯 크기

visionOS에서는 다음 위젯 패밀리를 지원합니다:

```swift
import WidgetKit
import SwiftUI

struct SpaceWidget: Widget {
    let kind = "SpaceWidget"
    
    var body: some WidgetConfiguration {
        StaticConfiguration(kind: kind, provider: SpaceProvider()) { entry in
            SpaceWidgetEntryView(entry: entry)
                .containerBackground(.fill.tertiary, for: .widget)
        }
        .configurationDisplayName("공간 위젯")
        .description("visionOS용 정보 위젯")
        .supportedFamilies([
            .systemSmall,
            .systemMedium,
            .systemLarge,
            .systemExtraLarge  // visionOS에서 사용 가능
        ])
    }
}
```

---

## 공간 디자인

### Glass Material 적용

visionOS 위젯은 공간 Glass 머티리얼 위에 표시됩니다.

```swift
struct SpaceWidgetEntryView: View {
    var entry: SpaceProvider.Entry
    
    var body: some View {
        VStack(alignment: .leading, spacing: 12) {
            // 위젯 헤더
            HStack {
                Image(systemName: "globe.americas.fill")
                    .font(.title2)
                    .foregroundStyle(.blue)
                Text("오늘의 세계")
                    .font(.headline)
            }
            
            // 콘텐츠
            Text(entry.headline)
                .font(.body)
                .lineLimit(3)
            
            Spacer()
            
            // 하단 정보
            Text(entry.date, style: .relative)
                .font(.caption)
                .foregroundStyle(.secondary)
        }
        .padding()
    }
}
```

---

## 인터랙티브 위젯

```swift
import AppIntents

// 인텐트 정의
struct PlayMusicIntent: AppIntent {
    static var title: LocalizedStringResource = "음악 재생"
    
    @Parameter(title: "트랙 ID")
    var trackID: String
    
    func perform() async throws -> some IntentResult {
        await MusicPlayer.shared.play(trackID: trackID)
        return .result()
    }
}

// 위젯 뷰
struct MusicWidgetView: View {
    var entry: MusicProvider.Entry
    
    var body: some View {
        VStack {
            // 앨범 아트
            if let artwork = entry.currentTrack?.artwork {
                Image(uiImage: artwork)
                    .resizable()
                    .aspectRatio(contentMode: .fit)
                    .clipShape(RoundedRectangle(cornerRadius: 12))
            }
            
            // 트랙 정보
            Text(entry.currentTrack?.title ?? "재생 중인 곡 없음")
                .font(.headline)
            
            Text(entry.currentTrack?.artist ?? "")
                .font(.subheadline)
                .foregroundStyle(.secondary)
            
            // 재생 컨트롤
            HStack(spacing: 24) {
                Button(intent: PlayMusicIntent(trackID: "prev")) {
                    Image(systemName: "backward.fill")
                }
                
                Button(intent: PlayMusicIntent(trackID: "toggle")) {
                    Image(systemName: entry.isPlaying 
                        ? "pause.fill" : "play.fill")
                        .font(.title)
                }
                
                Button(intent: PlayMusicIntent(trackID: "next")) {
                    Image(systemName: "forward.fill")
                }
            }
        }
        .padding()
    }
}
```

---

## visionOS 위젯 배치

### 공간 배치

```swift
// visionOS에서 위젯은 사용자의 공간에 자유롭게 배치 가능
// 시스템이 위치를 관리하며, 아래 속성으로 힌트를 제공

struct SpaceWidget: Widget {
    var body: some WidgetConfiguration {
        StaticConfiguration(kind: "space", provider: Provider()) { entry in
            EntryView(entry: entry)
        }
        .supportedFamilies([.systemMedium])
        // visionOS에서의 위젯 특성
        .contentMarginsDisabled()  // 마진 직접 관리 시
    }
}
```

---

## 멀티플랫폼 위젯

### iOS + visionOS 공유 위젯

```swift
struct MultiPlatformWidget: Widget {
    let kind = "MultiPlatform"
    
    var body: some WidgetConfiguration {
        StaticConfiguration(kind: kind, provider: Provider()) { entry in
            MultiPlatformEntryView(entry: entry)
                .containerBackground(.fill.tertiary, for: .widget)
        }
        .supportedFamilies(supportedFamilies)
    }
    
    // 플랫폼별 지원 크기
    var supportedFamilies: [WidgetFamily] {
        #if os(visionOS)
        return [.systemSmall, .systemMedium, .systemLarge, .systemExtraLarge]
        #else
        return [.systemSmall, .systemMedium, .systemLarge]
        #endif
    }
}

struct MultiPlatformEntryView: View {
    var entry: Provider.Entry
    @Environment(\.widgetFamily) var family
    
    var body: some View {
        switch family {
        case .systemSmall:
            CompactView(entry: entry)
        case .systemMedium:
            MediumView(entry: entry)
        case .systemLarge, .systemExtraLarge:
            DetailedView(entry: entry)
        default:
            CompactView(entry: entry)
        }
    }
}
```

---

## visionOS 위젯 디자인 가이드라인

### 공간 UI 원칙

1. **깊이 활용**: visionOS의 3D 공간에서 자연스럽게 보이도록
2. **적절한 크기**: 공간에서 편안하게 읽을 수 있는 텍스트 크기
3. **Glass 존중**: 시스템 Glass 효과를 가리지 않기
4. **시선 기반**: 사용자 시선에서 편안한 거리에 배치

### 접근성

```swift
// visionOS 위젯도 접근성 지원 필수
Text(entry.title)
    .font(.headline)
    .accessibilityLabel("현재 상태: \(entry.title)")
    .accessibilityAddTraits(.isHeader)
```

### 성능

- 위젯 업데이트는 타임라인 기반으로 효율적으로
- 공간에서 여러 위젯이 동시에 표시될 수 있으므로 리소스 절약
- 이미지는 적절한 크기로 리사이즈하여 제공

---

## Timeline Provider

```swift
struct SpaceProvider: TimelineProvider {
    func placeholder(in context: Context) -> SpaceEntry {
        SpaceEntry(date: Date(), title: "로딩 중...", subtitle: "")
    }
    
    func getSnapshot(in context: Context, completion: @escaping (SpaceEntry) -> Void) {
        let entry = SpaceEntry(
            date: Date(), 
            title: "오늘의 정보", 
            subtitle: "미리보기"
        )
        completion(entry)
    }
    
    func getTimeline(in context: Context, completion: @escaping (Timeline<SpaceEntry>) -> Void) {
        Task {
            let data = try await fetchLatestData()
            let entry = SpaceEntry(
                date: Date(), 
                title: data.title, 
                subtitle: data.subtitle
            )
            
            // 15분마다 업데이트
            let nextUpdate = Calendar.current.date(
                byAdding: .minute, value: 15, to: Date()
            )!
            
            let timeline = Timeline(
                entries: [entry], 
                policy: .after(nextUpdate)
            )
            completion(timeline)
        }
    }
}
```

---

## 관련 세션

- [Build widgets for the Shared Space (267)](https://developer.apple.com/videos/play/wwdc2025/267/)
